/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#pragma once

#include <chrono>
#include <condition_variable>
#include <mutex>
#include <thread>

class AutoResetEvent
{
private:
	bool flag_ = false;
	bool reset_ = false;
	bool interrupted_ = false;
	std::mutex protect_;
	std::condition_variable signal_;

public:
	explicit AutoResetEvent(bool initial = false)
	    : flag_(initial)
	{}
	AutoResetEvent(const AutoResetEvent&) = delete;
	AutoResetEvent& operator=(const AutoResetEvent&) = delete; // non-copyable

	void Set()
	{
		{
			std::lock_guard<std::mutex> _(protect_);
			reset_ = false;
			flag_ = true;
		}
		signal_.notify_all();
		std::this_thread::yield();
	}

	void Clear()
	{
		std::lock_guard<std::mutex> _(protect_);
		flag_ = false;
		reset_ = false;
		interrupted_ = false;
	}

	void Reset()
	{
		{
			std::lock_guard<std::mutex> _(protect_);
			reset_ = true;
			flag_ = false;
		}
		signal_.notify_all();
		std::this_thread::yield();
	}

	void Interrupt()
	{
		{
			std::lock_guard<std::mutex> _(protect_);
			interrupted_ = true;
		}
		signal_.notify_all();
		std::this_thread::yield();
	}

	bool WaitOne() { return WaitOne(-1); }

	bool WaitOne(long long timeout)
	{
		std::unique_lock<std::mutex> lk(protect_);
		bool ret = true;

		if (timeout == 0)
		{
			// if timeout is 0 then the caller merely wants to query the current status
			ret = flag_;
		}
		else
		{
			long long remainingTime = timeout;
			long long startTime = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count();

			while (!flag_ && !interrupted_ && !reset_)
			{
				// if timeout is negative, then we should wait forever until something happens
				// if timeout is positive, then we block until something happens or until the left-over time has elapsed
				if (timeout < 0)
				{
					signal_.wait(lk);
				}
				else
				{
					if ((remainingTime <= 0) || (signal_.wait_for(lk, std::chrono::milliseconds(remainingTime)) == std::cv_status::timeout))
					{
						ret = false;
						break;
					}
					remainingTime = timeout -
					    (std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count() - startTime);
				}
			}
		}

		if (interrupted_ || reset_)
			ret = false;

		flag_ = false;
		interrupted_ = false;
		reset_ = false;

		return ret;
	}
};